// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "exi_bba.h"
#include "degub.h"
#include "ini.h"
#include "common.h"
#include <Iphlpapi.h>
//#include <Ntddndis.h>

#define ARP_SRC_IP_OFFSET 0x1C
#define ARP_DST_IP_OFFSET 0x26
#define ARP_SRC_MAC_OFFSET 0x16
#define ARP_DST_MAC_OFFSET 0x20
#define IPv4_SRC_IP_OFFSET 0x1A
#define IPv4_DST_IP_OFFSET 0x1E

EXIDevice_BBA::EXIDevice_BBA(EXIInterruptRaiser& eir) : interrupt(eir),
mExpectVariableLengthImmWrite(false), mReadyToSend(false),
mHAdapter(INVALID_HANDLE_VALUE),
mHRecvEvent(INVALID_HANDLE_VALUE), mHReadWait(INVALID_HANDLE_VALUE),
mWriteBuffer(2*K), mCbw(mBbaMem + CB_OFFSET, CB_SIZE),
mWriteP(INVALID_P), mReadP(INVALID_P), mExpectSpecialImmRead(false),
mPacketsSent(0), mPacketsRcvd(0), mRecvBuffer(2048),
mWaiting(false), mRecvBufferLength(0), mRBEmpty(true) {
	memset(mBbaMem, 0, BBAMEM_SIZE);
}

//------------------------------private functions-------------------------------

#define BBA_INTERRUPT_RECV 0x02
#define BBA_INTERRUPT_SENT 0x04
#define BBA_INTERRUPT_RECV_ERROR 0x08
#define BBA_INTERRUPT_SEND_ERROR 0x10

EXIDevice_BBA::~EXIDevice_BBA() {
	if(isActivated()) {
		HWGLE(deactivate());
		DEGUB("%i packets sent, %i recieved\n", mPacketsSent, mPacketsRcvd);
	} else {
		DEGUB("The BBA was never activated.\n");
	}
}

#if 0
/*
* in_cksum --
*      Checksum routine for Internet Protocol family headers (C Version)
*/
unsigned short in_cksum(unsigned short *addr,int len)
{
	register int sum = 0;
	u_short answer = 0;
	register u_short *w = addr;
	register int nleft = len;

	/*
	* Our algorithm is simple, using a 32 bit accumulator (sum), we add
	* sequential 16 bit words to it, and at the end, fold back all the
	* carry bits from the top 16 bits into the lower 16 bits.
	*/
	while (nleft > 1)  {
		sum += *w++;
		nleft -= 2;
	}

	/* mop up an odd byte, if necessary */
	if (nleft == 1) {
		*(u_char *)(&answer) = *(u_char *)w ;
		sum += answer;
	}

	/* add back carry outs from top 16 bits to low 16 bits */
	sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
	sum += (sum >> 16);                     /* add carry */
	answer = (u_short)~sum;                          /* truncate to 16 bits */
	return(answer);
}

enum EtherType { ARPRequest, ARPResponse, IPv4, UnknownType=3 };

EtherType getEtherType(const BYTE *etherpckt, size_t size) {
	static const BYTE IPv4_bytes[3] = { 0x08, 0x00, 0x45 };
	static const BYTE ARP_bytes[9] = { 0x08, 0x06, 0x00, 0x01, 0x08,
		0x00, 0x06, 0x04, 0x00 };

	if(size >= 0xE + 20)	//(Ethernet + IP) headers
		if(memcmp(etherpckt + 0xC, IPv4_bytes, 3) == 0) {
			//BBADEGUB("IPv4\n");
			return IPv4;
		}

		if(size >= 42)  //Effective ARP size
			if(memcmp(etherpckt + 0xC, ARP_bytes, 9) == 0) {
				if(etherpckt[0x15] == 0x01) {
					//BBADEGUB("ARP Request\n");
					return ARPRequest;
				}
				if(etherpckt[0x15] == 0x02) {
					//BBADEGUB("ARP Response\n");
					return ARPResponse;
				}
			}

			return UnknownType;
}

bool checkIPchecksum(const BYTE *ipv4pckt, size_t size) {
	MYASSERT(size >= 20);
	Container<BYTE> temp(size);
	memcpy(temp, ipv4pckt, size);
	MAKE(WORD, temp[10]) = 0x0000;
	return MAKE(WORD, ipv4pckt[10]) == in_cksum((WORD*)temp.p(), size);
}

void recalculateIPchecksum(BYTE *ipv4pckt, size_t size) {
	MYASSERT(size >= 20);
	MAKE(WORD, ipv4pckt[10]) = 0x0000;
	MAKE(WORD, ipv4pckt[10]) = in_cksum((WORD*)ipv4pckt, size);
}

inline const char *wc_ntoa(IPAddr ip_address) {
	return inet_ntoa(MAKE(in_addr, ip_address));
}

bool IPCheckAndReplace(BYTE *etherpckt, size_t offset, IPAddr check, IPAddr replace) {
	if(MAKE(IPAddr, etherpckt[offset]) == check) {
		/*char od[16], nd[16];
		strcpy(od, wc_ntoa(check));
		strcpy(nd, wc_ntoa(replace));
		BBADEGUB("Changing IP: %s to %s\n", od, nd);*/

		MAKE(IPAddr, etherpckt[offset]) = replace;
		return true;
	} else
		return false;
}
#endif	//0

void PrintPacket(BYTE *buf, DWORD size)
{
	ULONG	i, j, ulLines, ulen;
	BYTE	*pChar, *pLine, *base;

	DEGUB("Packet length: %ld bytes\n", size);
	ulLines = (size + 15) / 16;

	pChar = buf;
	base = pChar;

	for(i=0; i<ulLines; i++) {
		pLine = pChar;
		DEGUB("%04lx : ", pChar - base);
		ulen = size;
		ulen = (ulen > 16) ? 16 : ulen;
		size -= ulen;
		for(j=0; j<ulen; j++) {
			if(j == 8) {
				DEGUB(" ");
			}
			DEGUB("%02x ", *(BYTE *)pChar++);
		}
		if(ulen < 16)
			DEGUB("%*s", (16-ulen)*3, " ");
		pChar = pLine;
		for(j=0; j<ulen; j++, pChar++) {
			if(j == 8) {
				DEGUB(" ");
			}
			DEGUB("%c", isprint(*pChar) ? *pChar : '.');
		}
		DEGUB("\n");
	} 
	DEGUB("\n");
}

bool EXIDevice_BBA::sendPacket(BYTE *etherpckt, size_t size) {
	if(g::bba_log) {
		BBADEGUB("sendPacket: ");
		PrintPacket(etherpckt, size);
	}
	DWORD numBytesWrit;
	OVERLAPPED overlap;
	ZERO_OBJECT(overlap);
	//overlap.hEvent = mHRecvEvent;
	TGLE(WriteFile(mHAdapter, etherpckt, size, &numBytesWrit, &overlap));
	if(numBytesWrit != size) {
		DEGUB("BBA sendPacket %i only got %i bytes sent!\n", size, numBytesWrit);
		FAIL(UE_BBA_ERROR);
	}
	recordSendComplete();
	return true;
}
void EXIDevice_BBA::recordSendComplete() {
	mBbaMem[0x00] &= ~0x06;
	if(mBbaMem[0x08] & BBA_INTERRUPT_SENT) {
		mBbaMem[0x09] |= BBA_INTERRUPT_SENT;
		BBADEGUB("BBA Send interrupt raised\n");
		interrupt.raiseEXI("BBA Send");
	}
	mPacketsSent++;
}

bool EXIDevice_BBA::checkRecvBuffer() {
	if(mRecvBufferLength != 0) {
		TGLE(handleRecvdPacket());
	}
	return true;
}

//Stores incoming packet in CB, updates p_write, but not mRRBPP
bool EXIDevice_BBA::handleRecvdPacket() {
	if(g::bba_log) {
		BBADEGUB("Handling packet, %i bytes\n", mRecvBufferLength);
		//PrintPacket(mRecvBuffer, mRecvBufferLength);
	}

	int rbwpp = mCbw.p_write() + CB_OFFSET;	//read buffer write page pointer
	DWORD available_bytes_in_cb;
	if(rbwpp < mRBRPP)
		available_bytes_in_cb = mRBRPP - rbwpp;
	else if(rbwpp == mRBRPP)
		available_bytes_in_cb = mRBEmpty ? CB_SIZE : 0;
	else //rbwpp > mRBRPP
		available_bytes_in_cb = CB_SIZE - rbwpp + (mRBRPP - CB_OFFSET);

	//DUMPWORD(rbwpp);
	//DUMPWORD(mRBRPP);
	//DUMPWORD(available_bytes_in_cb);

	MYASSERT(available_bytes_in_cb <= CB_SIZE);
	if(available_bytes_in_cb != CB_SIZE)//< mRecvBufferLength + SIZEOF_RECV_DESCRIPTOR)
		return true;
	TGLE(cbwriteDescriptor(mRecvBufferLength));
	mCbw.write(mRecvBuffer, mRecvBufferLength);
	mCbw.align();
	rbwpp = mCbw.p_write() + CB_OFFSET;
	//DUMPWORD(rbwpp);

	mPacketsRcvd++;
	mRecvBufferLength = 0;

	if(mBbaMem[0x08] & BBA_INTERRUPT_RECV) {
		if(!(mBbaMem[0x09] & BBA_INTERRUPT_RECV)) {
			mBbaMem[0x09] |= BBA_INTERRUPT_RECV;
			BBADEGUB("BBA Recv interrupt raised\n");
			interrupt.raiseEXI("BBA Recv");
		}
	}

	if(mBbaMem[BBA_NCRA] & BBA_NCRA_SR) {
		TGLE(startRecv());
	}

	return true;
}

union bba_descr {
	struct { u32 next_packet_ptr:12, packet_len:12, status:8; };
	u32 word;
};

bool EXIDevice_BBA::cbwriteDescriptor(DWORD size) {
	//if(size < 0x3C) {//60
#define ETHERNET_HEADER_SIZE 0xE
	if(size < ETHERNET_HEADER_SIZE) {
		DEGUB("Packet too small: %i bytes\n", size);
		FAIL(UE_BBA_ERROR);
	}

	size += SIZEOF_RECV_DESCRIPTOR;  //The descriptor supposed to include the size of itself

	//We should probably not implement wraparound here,
	//since neither tmbinc, riptool.dol, or libogc does...
	if(mCbw.p_write() + SIZEOF_RECV_DESCRIPTOR >= CB_SIZE) {
		DEGUB("The descriptor won't fit\n");
		FAIL(UE_BBA_ERROR);
	}
	if(size >= CB_SIZE) {
		DEGUB("Packet too big: %i bytes\n", size);
		FAIL(UE_BBA_ERROR);
	}

	bba_descr descr;
	descr.word = 0;
	descr.packet_len = size;
	descr.status = 0;
	DWORD npp;
	if(mCbw.p_write() + size < CB_SIZE) {
		npp = mCbw.p_write() + size + CB_OFFSET;
	} else {
		npp = mCbw.p_write() + size + CB_OFFSET - CB_SIZE;
	}
	npp = (npp + 0xff) & ~0xff;
	if(npp >= CB_SIZE + CB_OFFSET)
		npp -= CB_SIZE;
	descr.next_packet_ptr = npp >> 8;
	//DWORD swapped = swapw(descr.word);
	//next_packet_ptr:12, packet_len:12, status:8;
	BBADEGUB("Writing descriptor 0x%08X @ 0x%04X: next 0x%03X len 0x%03X status 0x%02X\n",
		descr.word, mCbw.p_write() + CB_OFFSET, descr.next_packet_ptr,
		descr.packet_len, descr.status);
	mCbw.write(&descr.word, SIZEOF_RECV_DESCRIPTOR);

	return true;
}

void EXIDevice_BBA::CyclicBufferWriter::write(void *src, size_t size) {
	MYASSERT(size < _cap);
	BYTE* bsrc = (BYTE*) src;
	if(_write + size >= _cap) { //wraparound
		memcpy(_buffer + _write, src, _cap - _write);
		memcpy(_buffer, bsrc + (_cap - _write), size - (_cap - _write));
		_write = size - (_cap - _write);
	} else {
		memcpy(_buffer + _write, src, size);
		_write += size;
	}
	//DEGUB("CBWrote %i bytes\n", size);
}
void EXIDevice_BBA::CyclicBufferWriter::align() {
	_write = (_write + 0xff) & ~0xff;
	if(_write >= _cap)
		_write -= _cap;
}

//------------------------------public functions--------------------------------

void EXIDevice_BBA::notify_deselected() {
	if(mExpectVariableLengthImmWrite) {
		BBADEGUB("Variable write complete. Final size: %i bytes\n",
			mWriteBuffer.size());
		mExpectVariableLengthImmWrite = false;

		mReadyToSend = true;
	}

	mWriteP = mReadP = INVALID_P;
	mExpectSpecialImmRead = false;
}

EXIResultValue EXIDevice_BBA::writeImm(DWORD size, DWORD data) {
	//BBADEGUB("BBA writeImm, %i byte%s: 0x%0*X\n",
	//size, (size==1?"":"s"), size*2, data);
	if(mExpectVariableLengthImmWrite) {
		if(size == 4)
			data = swapw(data);
		else if(size == 2)
			data = swaph((WORD)data);
		mWriteBuffer.write(size, &data);
		return EXI_HANDLED;
	} else if(mWriteP != INVALID_P) {
		if(mWriteP + size > BBAMEM_SIZE) {
			DEGUB("Write error: mWriteP + size = 0x%04X + %i\n", mWriteP, size);
			return EXI_UNHANDLED;
		}
		BBADEGUB("Write to BBA address 0x%0*X, %i byte%s: 0x%0*X\n",
			mWriteP >= CB_OFFSET ? 4 : 2, mWriteP, size, (size==1?"":"s"), size*2, data);

		switch(mWriteP) {
			case 0x09:
				BBADEGUB("BBA Interrupt reset 0x%02X & ~(0x%02X) => 0x%02X\n",
					mBbaMem[0x09], MAKE(BYTE, data), mBbaMem[0x09] & ~MAKE(BYTE, data));
				MYASSERT(size == 1);
				mBbaMem[0x09] &= ~MAKE(BYTE, data);
				break;
			case BBA_NCRA:
#define RISE(flags) ((data & (flags)) && !(mBbaMem[0x00] & (flags)))
				if(RISE(BBA_NCRA_RESET)) {
					BBADEGUB("BBA Reset\n");
				}
				if(RISE(BBA_NCRA_SR) && isActivated()) {
					BBADEGUB("BBA Start Recieve\n");
					HWGLE(startRecv());
				}
				if(RISE(BBA_NCRA_ST1)) {
					BBADEGUB("BBA Start Transmit\n");
					if(!mReadyToSend)
						throw hardware_fatal_exception("BBA Transmit without a packet!");
					HWGLE(sendPacket(mWriteBuffer.p(), mWriteBuffer.size()));
					mReadyToSend = false;
				}
				mBbaMem[0x00] = MAKE(BYTE, data);
				break;
			case BBA_NWAYC:
				if(data & (BBA_NWAYC_ANE | BBA_NWAYC_ANS_RA)) {
					HWGLE(activate());
					//say we've successfully negotiated for 10 Mbit full duplex
					//should placate libogc
					mBbaMem[BBA_NWAYS] = BBA_NWAYS_LS10 | BBA_NWAYS_LPNWAY |
						BBA_NWAYS_ANCLPT | BBA_NWAYS_10TXF;
				}
				break;
			case 0x18:	//RRP - Receive Buffer Read Page Pointer
				MYASSERT(size == 2 || size == 1);
				mRBRPP = (BYTE)data << 8;	//I hope this works with both write sizes.
				mRBEmpty = mRBRPP == ((WORD)mCbw.p_write() + CB_OFFSET);
				HWGLE(checkRecvBuffer());
				break;
			case 0x16:	//RWP
				MYASSERT(size == 2 || size == 1);
				MYASSERT(data == DWORD((WORD)mCbw.p_write() + CB_OFFSET) >> 8);
				break;
			default:
				memcpy(mBbaMem + mWriteP, &data, size);
				mWriteP = mWriteP + WORD(size);
		}
		return EXI_HANDLED;
	} else if(size == 2 && (WORD)data == 0) {	//Device ID request
		BBADEGUB("BBA Device ID request\n");
		mSpecialImmData = EXI_DEVTYPE_ETHER;
		mExpectSpecialImmRead = true;
		return EXI_HANDLED;
	} else if((size == 4 && (data & 0xC0000000) == 0xC0000000) ||
		(size == 2 && (data & 0x4000) == 0x4000)) { //Write to BBA register
			if(size == 4)
				mWriteP = (BYTE)getbitsw(data, 16, 23);
			else  //size == 2
				mWriteP = (BYTE)getbitsw(data & ~0x4000, 16, 23);  //Dunno about this...
			if(mWriteP == 0x48) {
				mWriteBuffer.clear();
				mExpectVariableLengthImmWrite = true;
				BBADEGUB("Prepared for variable length write to address 0x48\n");
			} else {
				//BBADEGUB("BBA Write pointer set to 0x%0*X\n", size, mWriteP);
			}
			return EXI_HANDLED;
	} else if((size == 4 && (data & 0xC0000000) == 0x80000000) ||
		(size == 2 && (data & 0x4000) == 0x0000)) //Read from BBA register
	{
		if(size == 4) {
			mReadP = (WORD)getbitsw(data, 8, 23);
			if(mReadP >= BBAMEM_SIZE) {
				DEGUB("Illegal BBA address: 0x%04X\n", mReadP);
				//if(g::bouehr)
				throw bouehr_exception("Illegal BBA address");
				//return EXI_UNHANDLED;
			}
		} else {  //size == 2
			mReadP = (BYTE)getbitsw(data, 16, 23);
		}
		switch(mReadP) {
		case 0x20:	//MAC address
			memcpy(mBbaMem + mReadP, g::mac_address, 6);
			break;
		case 0x01:	//Revision ID
			break;
		case 0x16:	//RWP - Receive Buffer Write Page Pointer
			MAKE(WORD, mBbaMem[mReadP]) = ((WORD)mCbw.p_write() + CB_OFFSET) >> 8;
			break;
		case 0x18:	//RRP - Receive Buffer Read Page Pointer
			MAKE(WORD, mBbaMem[mReadP]) = (mRBRPP) >> 8;
			break;
		case 0x3A:	//bit 1 set if no data available
			mBbaMem[mReadP] = !mRBEmpty;
			break;
		case 0x00:
			//mBbaMem[mReadP] = 0x00;
			//if(!sendInProgress())
			mBbaMem[mReadP] &= ~(0x06);
			break;
		case 0x03:
			mBbaMem[mReadP] = 0x80;
			break;
		}
		//BBADEGUB("BBA Read pointer set to 0x%0*X\n", size, mReadP);
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled BBA Imm write: %i, 0x%08X\n", size, data);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled BBA Imm write");
		//return EXI_UNHANDLED;
	}
}

EXIResultValue EXIDevice_BBA::readImm(DWORD size, DWORD &data) {
	if(mExpectSpecialImmRead) {
		data = mSpecialImmData;
		mExpectSpecialImmRead = false;
		return EXI_HANDLED;
	}
	if(mReadP != INVALID_P) {
		if(mReadP + size > BBAMEM_SIZE) {
			DEGUB("Read error: mReadP + size = 0x%04X + %i\n", mReadP, size);
			return EXI_UNHANDLED;
		}
		memcpy(&data, mBbaMem + mReadP, size);
		data = swapw(data); //we have a byteswap problem...
		BBADEGUB("Read from BBA address 0x%0*X, %i byte%s: 0x%0*X\n",
			mReadP >= CB_OFFSET ? 4 : 2, mReadP, size, (size==1?"":"s"),
			size*2, getbitsw(data, 0, size * 8 - 1));
		mReadP = mReadP + (WORD)size;
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled BBA Imm read: %i bytes\n", size);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled BBA Imm read");
		//return EXI_UNHANDLED;
	}
}

EXIResultValue EXIDevice_BBA::writeDMA(DWORD size, DWORD address, MemInterface &mem) {
	if(mExpectVariableLengthImmWrite) {
		mWriteBuffer.write(size, mem.getp_physical(address, size));
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled BBA DMA write: %i, 0x%08X\n", size, address);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled BBA DMA write");
		//return EXI_UNHANDLED;
	}
}

EXIResultValue EXIDevice_BBA::readDMA(DWORD size, DWORD address, MemInterface &mem) {
	if(mReadP != INVALID_P) {
		if(mReadP + size > BBAMEM_SIZE) {
			DEGUB("Read error: mReadP + size = 0x%04X + %i\n", mReadP, size);
			return EXI_UNHANDLED;
		}
		mem.write_physical(address, size, mBbaMem + mReadP);
		BBADEGUB("DMA Read from BBA address 0x%0*X, %i bytes\n",
			mReadP >= CB_OFFSET ? 4 : 2, mReadP, size);
		mReadP = mReadP + (WORD)size;
		return EXI_HANDLED;
	} else {
		DEGUB("Unhandled BBA DMA read: %i, 0x%08X\n", size, address);
		//if(g::bouehr)
		throw bouehr_exception("Unhandled BBA DMA read");
		//return EXI_UNHANDLED;
	}
}
